<?php

namespace App\Http\Controllers;

use App\Models\Doctor;
use App\Models\Appointment;
use App\Models\Prescription;
use App\Models\DoctorAvailability;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;

class DoctorController extends Controller
{
    public function dashboard()
    {
        $doctor = Auth::user()->doctor;
        $user = Auth::user();
        
        $stats = [
            'appointments' => Appointment::where('doctor_id', $doctor->id)->count(),
            'pending_appointments' => Appointment::where('doctor_id', $doctor->id)->where('status', 'pending')->count(),
            'prescriptions' => Prescription::where('doctor_id', $doctor->id)->count(),
        ];

        $recentAppointments = Appointment::where('doctor_id', $doctor->id)
            ->with(['patient', 'branch', 'pharmacist'])
            ->latest()
            ->take(10)
            ->get();

        // Get unread notifications
        $notifications = $user->unreadNotifications()->latest()->take(10)->get();

        return view('doctor.dashboard', compact('stats', 'recentAppointments', 'notifications'));
    }

    public function profile()
    {
        $doctor = Auth::user()->doctor;
        return view('doctor.profile', compact('doctor'));
    }

    public function updateProfile(Request $request)
    {
        $doctor = Auth::user()->doctor;

        $validated = $request->validate([
            'title' => 'nullable|string',
            'first_name' => 'required|string',
            'last_name' => 'required|string',
            'specialization' => 'nullable|string',
            'qualification' => 'nullable|string',
            'practice_number' => 'required|string|unique:doctors,practice_number,' . $doctor->id,
            'country' => 'nullable|string',
            'educational_details' => 'nullable|string',
            'phone' => 'nullable|string',
            'profile_picture' => 'nullable|image|max:2048',
            'letterhead' => 'nullable|image|max:2048',
            'stamp' => 'nullable|image|max:2048',
            'signature' => 'nullable|image|max:2048',
        ]);

        if ($request->hasFile('profile_picture')) {
            if ($doctor->profile_picture_path) {
                Storage::disk('public')->delete($doctor->profile_picture_path);
            }
            $validated['profile_picture_path'] = $request->file('profile_picture')->store('doctors/profile-pictures', 'public');
        }

        if ($request->hasFile('letterhead')) {
            if ($doctor->letterhead_path) {
                Storage::disk('public')->delete($doctor->letterhead_path);
            }
            $validated['letterhead_path'] = $request->file('letterhead')->store('doctors/letterheads', 'public');
        }

        if ($request->hasFile('stamp')) {
            if ($doctor->stamp_path) {
                Storage::disk('public')->delete($doctor->stamp_path);
            }
            $validated['stamp_path'] = $request->file('stamp')->store('doctors/stamps', 'public');
        }

        if ($request->hasFile('signature')) {
            if ($doctor->signature_path) {
                Storage::disk('public')->delete($doctor->signature_path);
            }
            $validated['signature_path'] = $request->file('signature')->store('doctors/signatures', 'public');
        }

        unset($validated['profile_picture'], $validated['letterhead'], $validated['stamp'], $validated['signature']);

        $doctor->update($validated);

        return back()->with('success', 'Profile updated successfully.');
    }

    public function appointments()
    {
        $doctor = Auth::user()->doctor;
        $appointments = Appointment::where('doctor_id', $doctor->id)
            ->with(['patient', 'branch', 'pharmacist', 'vitals'])
            ->latest()
            ->get();

        return view('doctor.appointments', compact('appointments'));
    }

    public function prescriptions()
    {
        $doctor = Auth::user()->doctor;
        $prescriptions = Prescription::where('doctor_id', $doctor->id)
            ->with(['patient', 'appointment', 'items'])
            ->latest()
            ->get();

        return view('doctor.prescriptions', compact('prescriptions'));
    }

    public function videoCall($id)
    {
        $appointment = Appointment::with(['patient', 'doctor'])->findOrFail($id);
        $doctor = Auth::user()->doctor;
        
        if ($appointment->doctor_id !== $doctor->id) {
            abort(403, 'Unauthorized access to this appointment.');
        }

        return view('doctor.video-call', compact('appointment'));
    }

    public function markNotificationRead($id)
    {
        $notification = Auth::user()->notifications()->findOrFail($id);
        $notification->markAsRead();
        
        return back()->with('success', 'Notification marked as read.');
    }

    public function markAllNotificationsRead()
    {
        Auth::user()->unreadNotifications->markAsRead();
        
        return back()->with('success', 'All notifications marked as read.');
    }

    public function notifications()
    {
        $user = Auth::user();
        $notifications = $user->notifications()->latest()->paginate(20);
        return view('doctor.notifications', compact('notifications'));
    }

    public function availability()
    {
        $doctor = Auth::user()->doctor;
        $availabilities = DoctorAvailability::where('doctor_id', $doctor->id)
            ->orderByRaw("FIELD(day_of_week, 'monday', 'tuesday', 'wednesday', 'thursday', 'friday', 'saturday', 'sunday')")
            ->orderBy('start_time')
            ->get();
        
        return view('doctor.availability', compact('availabilities'));
    }

    public function storeAvailability(Request $request)
    {
        $doctor = Auth::user()->doctor;

        $validated = $request->validate([
            'day_of_week' => 'required|in:monday,tuesday,wednesday,thursday,friday,saturday,sunday',
            'start_time' => 'required|date_format:H:i',
            'end_time' => 'required|date_format:H:i|after:start_time',
            'is_available' => 'nullable|boolean',
        ]);

        DoctorAvailability::updateOrCreate(
            [
                'doctor_id' => $doctor->id,
                'day_of_week' => $validated['day_of_week'],
                'start_time' => $validated['start_time'],
                'end_time' => $validated['end_time'],
            ],
            [
                'is_available' => $validated['is_available'] ?? true,
            ]
        );

        return back()->with('success', 'Availability schedule updated successfully.');
    }

    public function deleteAvailability($id)
    {
        $doctor = Auth::user()->doctor;
        $availability = DoctorAvailability::where('doctor_id', $doctor->id)->findOrFail($id);
        $availability->delete();

        return back()->with('success', 'Availability slot deleted successfully.');
    }
}

